async function prepareForPrint() {
    const input = document.getElementById('imageInput');
    const titleInput = document.getElementById('titleInput');
    const printArea = document.getElementById('printArea');
    printArea.innerHTML = '';

    const files = Array.from(input.files);
    if (files.length === 0) {
        alert('PNG画像またはZIPファイルを選択してください。');
        return;
    }

    const numberRegex = /(\d+)\.png$/;
    let numberedFiles = [];

	// --- ファイル読み込み（PNGまたはZIP） ---
	for (const file of files) {
	    const name = file.name.toLowerCase();

	    if (name.endsWith('.zip')) {
	        try {
	            const zip = await JSZip.loadAsync(file);
	            const entries = Object.values(zip.files);

	            for (const entry of entries) {
	                // ✅ フォルダは除外
	                if (entry.dir) continue;

	                // ✅ PNG拡張子を確認（大文字・小文字混在対応）
	                if (!entry.name.toLowerCase().endsWith('.png')) continue;

	                // ✅ フォルダ構造対応＆qr_00000.pngなどの番号を抽出
	                const match = entry.name.match(/(\d+)\.png$/i);
	                const number = match ? parseInt(match[1], 10) : -1;
	                if (number < 0) continue;

	                // Blob化
	                const blob = await entry.async('blob');

	                numberedFiles.push({
	                    name: entry.name,
	                    number,
	                    file: new File([blob], entry.name, { type: 'image/png' })
	                });
	            }
	        } catch (err) {
	            console.error('ZIP展開エラー:', err);
	            alert('ZIPファイルの読み込みに失敗しました。パスワードなしZIPを使用してください。');
	            return;
	        }
	    } else if (name.endsWith('.png')) {
	        const match = file.name.match(/(\d+)\.png$/i);
	        const number = match ? parseInt(match[1], 10) : -1;
	        if (number >= 0) {
	            numberedFiles.push({ file, number, name: file.name });
	        }
	    }
	}

	// --- ZIPもPNGも見つからなかった場合 ---
	if (numberedFiles.length === 0) {
	    alert('PNG画像が見つかりません。');
	    return;
	}


    // --- 並び替え（ヘッダーQRを最初に） ---
    numberedFiles.sort((a, b) => {
        if (a.number === 0) return -1;
        if (b.number === 0) return 1;
        return a.number - b.number;
    });

    const maxNumber = Math.max(...numberedFiles.map(f => f.number));
    const totalPages = Math.ceil(numberedFiles.length / 20);
    const titleText = titleInput.value.trim() || 'QRコード一覧';

    // --- 画像をすべて読み込んでから描画 ---
    const images = await Promise.all(
        numberedFiles.map(item => {
            return new Promise(resolve => {
                const reader = new FileReader();
                reader.onload = e => {
                    resolve({
                        number: item.number,
                        name: item.name,
                        src: e.target.result
                    });
                };
                reader.readAsDataURL(item.file);
            });
        })
    );

    for (let pageIndex = 0; pageIndex < totalPages; pageIndex++) {
        const start = pageIndex * 20;
        const end = start + 20;
        const chunk = images.slice(start, end);

        // ページコンテナ
        const pageDiv = document.createElement('div');
        pageDiv.className = 'print-page';
        if (pageIndex > 0) pageDiv.style.pageBreakBefore = 'always';

        // ヘッダー
        const headerDiv = document.createElement('div');
        headerDiv.className = 'print-header';
        headerDiv.innerHTML = `
            <span class="print-title">${titleText}</span>
            <span class="print-page-number">${pageIndex + 1} / ${totalPages} ページ</span>
        `;
        pageDiv.appendChild(headerDiv);

        // グリッド
        const grid = document.createElement('div');
        grid.className = 'print-grid';
        chunk.forEach(item => {
            const imgContainer = document.createElement('div');
            imgContainer.className = 'image-item';

            const img = document.createElement('img');
            img.src = item.src;
            img.alt = item.name;

            const caption = document.createElement('div');
            caption.className = 'image-caption';
            caption.textContent =
                item.number === 0
                    ? 'ヘッダーQR'
                    : `分割QR #${item.number}/${maxNumber}`;

            imgContainer.appendChild(img);
            imgContainer.appendChild(caption);
            grid.appendChild(imgContainer);
        });

        pageDiv.appendChild(grid);
        printArea.appendChild(pageDiv);
    }

    // --- 印刷ボタンを有効化 ---
    document.dispatchEvent(new Event('printReady'));
}

// 🔽 印刷準備完了後に印刷ボタンを有効化
document.addEventListener('printReady', () => {
    const printButton = document.getElementById('printButton');
    if (printButton) printButton.disabled = false;
});
