/**
 * スクリプト名: script.js
 * 機能: バイナリファイルをBase64にエンコード後、指定文字数で分割し、連番付きの複数のQRコードを生成し、クリックで拡大・ナビゲーション表示する。
 * 備考: オフライン動作のため、qrious.min.js, jszip.min.js, FileSaver.min.jsがindex.htmlと同じフォルダにある必要があります。
 */

// グローバル状態管理
const qrDataStore = {}; // QRコードのデータ（連番、ECL、Base64値）を保存
const generatedCanvases = []; // ダウンロード用に生成されたすべてのCanvas要素を格納
let currentChunkIndex = 0; // 現在モーダルで表示中のチャンク番号

// DOM要素の取得
const progressBarContainer = document.getElementById('progressBarContainer');
const progressBar = document.getElementById('progressBar');
const generateButton = document.getElementById('generateButton');
const downloadButton = document.getElementById('downloadButton');

// ----------------------
// プログレスバー制御関数
// ----------------------

function updateProgress(percentage, text) {
    progressBarContainer.style.display = 'block';
    progressBar.style.width = `${percentage}%`;
    progressBar.textContent = text || `${Math.round(percentage)}%`;
}

function resetProgress() {
    progressBarContainer.style.display = 'none';
    progressBar.style.width = '0%';
    progressBar.textContent = '0%';
}

// ----------------------
// データ処理関数
// ----------------------

// データを指定されたサイズで分割する関数
function chunkData(data, size) {
    const chunks = [];
    for (let i = 0; i < data.length; i += size) {
        chunks.push(data.substring(i, i + size));
    }
    return chunks;
}

// ----------------------
// ユーティリティ関数（追加）
// ----------------------
/**
 * UTF-8文字列をBase64文字列に変換する
 * (日本語などのマルチバイト文字を安全にQRコードデータに格納するために使用)
 */
function stringToBase64(str) {
    // 日本語対応: UTF-8 -> URLエンコード -> バイナリ文字列 -> Base64
    try {
        return btoa(unescape(encodeURIComponent(str)));
    } catch (e) {
        console.error("Base64エンコードエラー。", e);
        return btoa(str);
    }
}

// ----------------------
// ダウンロード機能
// ----------------------

function downloadAllQRCodesAsZip() {
    if (generatedCanvases.length === 0) {
        alert('生成されたQRコードがありません。');
        return;
    }
    
    // ZIPライブラリがロードされているか確認
    if (typeof JSZip === 'undefined' || typeof saveAs === 'undefined') {
        alert('ダウンロード機能に必要なライブラリが読み込まれていません。jszip.min.jsとFileSaver.min.jsがindex.htmlと同じフォルダに配置されているか確認してください。');
        return;
    }

    // ダウンロード中はボタンを無効化
    downloadButton.disabled = true;
    downloadButton.textContent = 'ZIPファイル作成中...';

    const zip = new JSZip();
    const fileNameBase = document.getElementById('fileInput').files[0]?.name.split('.')[0] || 'qr_file';

    // すべてのキャンバスをPNG形式でZIPに追加
    generatedCanvases.forEach((canvas, index) => {
        // dataURLからBase64データ部分を抽出
        const dataURL = canvas.toDataURL('image/png');
        const base64Data = dataURL.replace(/^data:image\/(png|jpg);base64,/, '');
        
        const chunkNumber = index + 1;
        const fileName = `${fileNameBase}_chunk_${chunkNumber}_of_${generatedCanvases.length}.png`;
        
        // ZIPファイルに追加
        zip.file(fileName, base64Data, { base64: true });
    });

    // ZIPファイルを生成してダウンロード
    zip.generateAsync({ type: 'blob' })
        .then(function(content) {
            saveAs(content, `${fileNameBase}_QR_chunks.zip`);
            alert('ZIPファイルのダウンロードが完了しました。');
        })
        .catch(function(error) {
            console.error('ZIPファイルの生成中にエラーが発生しました:', error);
            alert('ZIPファイルの生成中にエラーが発生しました。コンソールを確認してください。');
        })
        .finally(() => {
            // 終了後にボタンを元に戻す
            downloadButton.disabled = false;
            downloadButton.textContent = '全QRコードをZIPダウンロード';
        });
}


// ----------------------
// モーダル・ナビゲーション関数
// ----------------------

// ナビゲーションボタンの状態を更新する関数
function updateNavButtons(totalChunks) {
    const prevButton = document.getElementById('prevButton');
    const nextButton = document.getElementById('nextButton');
    
    prevButton.disabled = currentChunkIndex <= 1;
    nextButton.disabled = currentChunkIndex >= totalChunks;
}

// 拡大表示用のQRコードをレンダリングする関数
function renderExpandedQr(chunkNumber) {
    const data = qrDataStore[chunkNumber];
    if (!data) return;
    
    // QRiousがロードされているか確認
    if (typeof QRious === 'undefined') {
        console.error("QRiousライブラリが読み込まれていません。");
        alert("QRiousライブラリが読み込まれていません。qrious.min.jsが配置されているか確認してください。");
        return;
    }

    const expandedCanvas = document.getElementById('expandedQrCanvas');
    const modalLabel = document.getElementById('modalLabel');
    const errorLevel = document.getElementById('errorLevel').value;
    
    // 現在のインデックスを更新
    currentChunkIndex = chunkNumber;

    // ラベルの更新
    modalLabel.textContent = `チャンク ${chunkNumber}/${data.totalChunks} (ECL: ${errorLevel})`;
    
    // 内部描画サイズを900に設定 (CSS側で画面にフィットさせる)
    expandedCanvas.width = 900; 
    expandedCanvas.height = 900; 

    // 拡大表示用のQRコードを再生成
    new QRious({
        element: expandedCanvas,
        value: data.encodedValue,
        size: 900, 
        level: errorLevel 
    });

    // ナビゲーションボタンの状態を更新
    updateNavButtons(data.totalChunks);
}

// 前後ボタンでチャンクを移動する関数
function navigateToChunk(direction) {
    const totalChunks = qrDataStore[1]?.totalChunks || 0; 
    const newIndex = currentChunkIndex + direction;

    if (newIndex >= 1 && newIndex <= totalChunks) {
        renderExpandedQr(newIndex);
    }
}

// 拡大表示用のモーダルを開く関数
function openModal(index) {
    if (!qrDataStore[index]) return;

    const modalOverlay = document.getElementById('modalOverlay');
    
    // 最初にキャンバスの描画とナビゲーションの初期設定を行う
    renderExpandedQr(index);
    
    modalOverlay.style.display = 'flex';
}

// モーダルを閉じる関数
function closeModal() {
    document.getElementById('modalOverlay').style.display = 'none';
    currentChunkIndex = 0; // インデックスをリセット
}


// ----------------------
// メイン生成関数
// ----------------------

// Base64に変換された文字列をQRコードとして生成する関数
function generateQRCodes(base64String) {
    const container = document.getElementById('qrContainer');
    const chunkSize = parseInt(document.getElementById('chunkSize').value, 10);
    const errorLevel = document.getElementById('errorLevel').value;
    
    container.innerHTML = '';
    Object.keys(qrDataStore).forEach(key => delete qrDataStore[key]);
    generatedCanvases.length = 0; // キャンバスリストをクリア

    if (!base64String) {
        alert('エンコードされたデータがありません。');
        resetProgress();
        return;
    }
    if (isNaN(chunkSize) || chunkSize < 100) {
        alert('有効な分割サイズ（100以上の数値）を入力してください。');
        resetProgress();
        return;
    }
    
    // Base64文字列の分割
    const dataChunks = chunkData(base64String, chunkSize);
    const totalChunks = dataChunks.length;
    
    const fileInput = document.getElementById('fileInput');
    const file = fileInput.files[0];
    
    // 【修正箇所】日本語ファイル名対応のため、ファイル名をBase64エンコードする
    const encodedFileName = stringToBase64(file.name); //
    
    // ヘッダーフォーマット: FILESTART|mimeType|encodedFileName|totalChunks|
    // ファイル名部分をエンコードされたものに置き換える
    const header = `FILESTART|${file.type}|${encodedFileName}|${totalChunks}|`; //
    
    const totalSizeKB = (base64String.length / 1024).toFixed(2);
        
    console.log("-----------------------------------------");
    console.log(`ファイルサイズ (Base64): ${totalSizeKB} KB`);
    console.log(`分割数: ${totalChunks}`);
    
    // 分割されたデータごとにQRコードを生成
    dataChunks.forEach((chunk, index) => {
        const chunkNumber = index + 1;
        
        let encodedValue = chunk;
        
        if (chunkNumber === 1) {
            encodedValue = header + chunk;
        } else {
            encodedValue = `CHUNK|${chunkNumber}/${totalChunks}|${chunk}`;
        }

        // データの保存
        qrDataStore[chunkNumber] = { encodedValue: encodedValue, totalChunks: totalChunks };
        
        // 1. QRコード表示用の要素 (<div> と <canvas>) を作成
        const itemDiv = document.createElement('div');
        itemDiv.className = 'qr-item';
        
        const label = document.createElement('p');
        label.innerHTML = `チャンク ${chunkNumber}/${totalChunks} (ECL: ${errorLevel})<br>
                           <span style="font-size: 0.8em; color: #cc3333;">(データサイズ: ${chunk.length}文字)</span>`;
        itemDiv.appendChild(label);
        
        const canvas = document.createElement('canvas');
        // ★修正点：内部描画サイズを300に統一 (CSSの300pxと一致)
        canvas.width = 300; 
        canvas.height = 300; 
        itemDiv.appendChild(canvas);

        // クリックイベントを追加: 拡大モーダルを開く
        itemDiv.addEventListener('click', () => openModal(chunkNumber));

        container.appendChild(itemDiv);
        generatedCanvases.push(canvas); // ダウンロード用にキャンバスを保存

        // 2. QRiousを使用してQRコードを生成
        new QRious({
            element: canvas,
            value: encodedValue,
            // ★修正点：QRiousの描画サイズも300に統一
            size: 300, 
            level: errorLevel 
        });
        
        // プログレスバーの更新 (生成フェーズ)
        const progress = (chunkNumber / totalChunks) * 100;
        updateProgress(progress, `生成中... ${chunkNumber}/${totalChunks}`);
    });
    
    alert(`QRコードの生成が完了しました。\n全 ${totalChunks} 個のコードが生成されました。`);
    resetProgress(); // 完了したらリセット
    downloadButton.disabled = false; // ダウンロードボタンを有効化
}

/**
 * ファイルの読み込みとBase64エンコードを行うメインハンドラ
 */
function handleFileAndGenerate() {
    const fileInput = document.getElementById('fileInput');
    
    console.log("--- handleFileAndGenerate 関数開始 ---");
    generateButton.disabled = true; // 生成中はボタンを無効化
    downloadButton.disabled = true; // ダウンロードボタンを無効化
    
    if (fileInput.files.length === 0) {
        alert('ファイルを選択してください。');
        generateButton.disabled = false;
        return;
    }
    
    const file = fileInput.files[0];
    const reader = new FileReader();

    // ファイル読み込みの進捗イベント
    reader.onprogress = function(event) {
        if (event.lengthComputable) {
            const progress = (event.loaded / event.total) * 100;
            updateProgress(progress, `ファイルを読み込み中... ${Math.round(progress)}%`);
        }
    };

    reader.onload = function(event) {
        console.log("--- ファイル読み込み (reader.onload) 完了 ---");
        const dataUri = event.target.result;
        const base64String = dataUri.split(',')[1];
        
        // プログレスバーをリセットして次のフェーズへ
        updateProgress(0, 'QRコード生成準備中...');

        generateQRCodes(base64String);
        generateButton.disabled = false; // 終了後にボタンを有効化
    };

    reader.onerror = function(event) {
        console.error("ファイル読み込みエラー:", event.target.error);
        alert('ファイルの読み込み中にエラーが発生しました。');
        generateButton.disabled = false;
        resetProgress();
    };

    console.log("--- ファイル読み込み (readAsDataURL) 呼び出し開始 ---");
    reader.readAsDataURL(file);
}

// ----------------------
// イベントリスナーの登録
// ----------------------

document.addEventListener('DOMContentLoaded', () => {
    // 1. 生成ボタンのイベントリスナー登録
    generateButton.addEventListener('click', handleFileAndGenerate);
    // 2. ダウンロードボタンのイベントリスナー登録
    downloadButton.addEventListener('click', downloadAllQRCodesAsZip);

    // 3. モーダルを閉じるイベントリスナー登録 (×ボタン, オーバーレイ, Escキー)
    document.getElementById('modalClose').addEventListener('click', closeModal);
    document.getElementById('modalOverlay').addEventListener('click', (e) => {
        if (e.target.id === 'modalOverlay') {
            closeModal();
        }
    });

    document.addEventListener('keydown', (e) => {
        const modalOverlay = document.getElementById('modalOverlay');
        const isModalOpen = modalOverlay.style.display === 'flex';

        if (e.key === 'Escape' && isModalOpen) {
            closeModal();
            return;
        }

        if (isModalOpen) {
            if (e.key === 'ArrowLeft') {
                e.preventDefault(); 
                navigateToChunk(-1);
            } else if (e.key === 'ArrowRight') {
                e.preventDefault();
                navigateToChunk(1);
            }
        }
    });

    // 4. ナビゲーションボタンのイベントリスナー登録
    document.getElementById('prevButton').addEventListener('click', () => navigateToChunk(-1));
    document.getElementById('nextButton').addEventListener('click', () => navigateToChunk(1));
});
